<?php
namespace wcf\system\wow;
use wcf\util\HTTPTools;
use wcf\system\exception\SystemException;
use wcf\system\WCF;
use wcf\util\StringUtil;
/**
 * The WoWCalender class reads claneder from battle.net.
 * extens class BNETSite and requires class BNETAsset
 * 
 * @author	Veneanar Falkenbann
 * @copyright	2015 Sylvanas Garde
 * @package	com.sylvanasgarde.guildtool
 * @category	Guild Tool
 */
class BNETCalendar extends BNETSite {

    /**
     * DATE_TIME_FORMAT
     * Datetime format see:
     * http://php.net/manual/en/function.date.php
     * default set to German format
     * @var	string
     */
    const DATE_TIME_FORMAT = 'd/m/Y H:i';

    /**
     * DATE_TIME_FORMAT
     * regex to parse Date/Time from events.
     * default set to German format
     * @var	string
     */
    const DATE_TIME_REGEX =  '/\d\d\/\d\d\/\d\d\d\d.*\d\d:\d\d/';

    const DATE_TIME_BNET_TIMEZONE = 'Europe/Berlin';

    /**
     * HTMNL Document for crawling
     * @var	object
     */
    protected $html;

    protected $eventlist = null;

    /**
     * Charname of the crawler
     * @var	string
     */
    protected $ownchar;

	/**
     * Opens a calendar site.
     * Requires login. 
     * charname is needed to check if the correct calendar site is open
     * you should login manually to set the default character in battle.net
     *  
     * @param	string		$bnetemail
     * @param	string		$password
     * @param	string		$charname
     */
	public function __construct($bnetemail, $password, $charname) {
        parent::__construct('https://'.GUILDTOOL_BNET_HOST.'/wow/en/vault/character/event', true, array('username'=>$bnetemail, 'password'=>$password));
        if (strpos($this->content, $charname) === false) {
            throw new SystemException('Calendar inaccessible. Maybe you should change the standart char.');
        }
        $this->ownchar=$charname;
        $this->html = new \DOMDocument('1.0', 'utf-8');
        @$this->html->loadHTML('<?xml encoding="UTF-8">' . $this->content . '</xml>');
    }

    public function getEventbyID($id) {
        $workarray = empty($this->eventlist) ? $this->getEventList() : $this->eventlist;
        foreach($workarray as $event) {
            if ($event['ID'] == $id) return $event;
        }
        return null;
    }

    public function getEventListbyName($name) {
        $workarray = empty($this->eventlist) ? $this->getEventList() : $this->eventlist;
        $retunrarray = array();
        foreach($workarray as $event) {
            if ($event['name'] == $name) $retunrarray[] = $event;
        }
        return $retunrarray;
    }

    /**
     * retrives the guildevents and personalevents for the character.
     *  
     * @return array
     */
    public function getEventList() {
        $HTMLeventList = $this->html->getElementById('event-list')->getElementsByTagName('ul')->item(0)->getElementsByTagName('li');
        $returnArray = array();
        $eventDateTime = '';
        foreach ($HTMLeventList as $HTMLevent) {
            if ($HTMLevent->hasAttributes()) {
                foreach ($HTMLevent->attributes as $a) {
                    if ($a->nodeName == "data-id") {
                        $checkdate = $HTMLevent->getElementsByTagName('span');
                        foreach($checkdate as $node) {
                            $classes = $node->getAttribute('class');
                            if (strpos($classes, 'datetime')> -1) {
                                $eventDateTime = $node->nodeValue;
                            }
                        }
                        $bnetTimeZone = new \DateTimeZone(self::DATE_TIME_BNET_TIMEZONE);
                        $eventTime = \DateTime::createFromFormat(self::DATE_TIME_FORMAT , $eventDateTime, $bnetTimeZone );
                        $eventTime->setTimezone(new \DateTimeZone('UTC'));
                        // DateTimeZone
                        $iconurl = $HTMLevent->getElementsByTagName('img')->item(0)->getAttribute('src');
                        $returnArray[] = array(
                            'ID'=>intval($a->nodeValue),
                            'name'=> $HTMLevent->getElementsByTagName('h4')->item(0)->nodeValue,
                            'datetime' => $eventTime->getTimestamp(),
                            'icon' => GUILDTOOL_BNET_API_SAVEASSET ? BNETAsset::download($iconurl) : $iconurl
                        );
                    }
                }
            }
        }
        $this->eventlist =  $returnArray;
        return $returnArray;
    }

    /**
     * retrives details of an event
     *  
     * @param array   EventID
     * @return array
     */
    public function getEventDetails($event) {
        $retrunArray = array(
            'ID'=> intval($event['ID']),
            'name'=> $event['name'],
            'datetime' => $event['datetime'],
            'description' => '',
            'attendees'=> array(),
            'icon' => $event['icon']
            );
        $this->url = 'https://'.GUILDTOOL_BNET_HOST.'/wow/en/vault/character/event/details?eventId=' . $event['ID'];
        self::openURL();
		if (strpos($this->content,'<h4 class="subheader name">' . $event['name']) === false) {
            // Exception
        }
        $eventHTML = new \DOMDocument('2.0', 'UTF-8');
        @$eventHTML->loadHTML('<?xml encoding="UTF-8">' . $this->content . '</xml>');
        foreach ($eventHTML->getElementsByTagName('div') as $checkDiv) {
            if (strpos($checkDiv->getAttribute('class'), 'description') !== false) {
                $retrunArray['description'] = StringUtil::trim($checkDiv->nodeValue);
            }
        }
        $attendesHTML = $eventHTML->getElementsByTagName('ol');
        foreach ($attendesHTML->item(0)->getElementsByTagName('li') as $attendeHTML) {
			$parsedAttende = mb_split ('\s+', StringUtil::trim( $attendeHTML->nodeValue));
            $attendeeName = $parsedAttende[0];
            $attendeeState = StringUtil::toLowerCase($parsedAttende[1]); 
			$state = 0;
            switch ($attendeeState) {
				case 'confirmed':
					$state = 1;
					break;
				case 'signed':
					$state = 2;
					break;
				case 'accepted':
					$state = 2;
					break;
				case 'stand':
					$state = 3;
					break;
				case 'tentative':
					$state = 4;
					break;
				case 'out':
					$state = 5;
					break;	
				case 'declined':
					$state = 5;
					break;
				case 'invited':
					$state = 6;
					break;
			}
			if ($this->ownchar !=$attendeeName) $retrunArray['attendees'][] = array('name'=>$attendeeName, 'state'=>$state, 'debug'=>$attendeeState);
        }
        return $retrunArray;
    }
}
