<?php
namespace wcf\system\wow;
use wcf\util\HTTPTools;
use wcf\system\exception\SystemException;
use wcf\system\WCF;
use wcf\util\StringUtil;


/**
 * The BNETAsset class downloads assets from battle.net if needed.
 * due legal reasons you should use it ONLY for blizzard assets.
 * Please read: http://us.blizzard.com/en-us/company/about/legal-faq.html
 * 
 * Do not forget a copyright reference if you use any blizzard content:
 * http://us.blizzard.com/en-us/company/about/copyrightnotices.html
 * 
 * @author	    Veneanar
 * @copyright	2015 Sylvanas Garde
 * @package	    com.sylvanasgarde.guildtool
 * @category    Dataparser
 */

class BNETAsset extends HTTPTools {

    /**
     * LOCAL_ASSET_DIR
     * place where to store the assetfiles
     * @var	string
     */
    const LOCAL_ASSET_DIR = '/images/bnet/';

    /**
     * $allowed_assets
     * Array with allowed file extensions
     * @var	array
     */
    private static $allowed_assets = array('jpg', 'jpeg', 'png', 'gif');
    
    /**
     * $allowd_orign_hosts
     * Array with allowed hosts without protocoll
     * e.g. media.blizzard.com NOT http://media.blizzard.com
     * @var	array
     */
    private static $allowd_orign_hosts = array('eu.battle.net', 'us.battle.net', 'tw.battle.net', 'kr.battle.net', 'media.blizzard.com');

    /**
     * check if the host is allowed. use without protocoll.
     * e.g. media.blizzard.com NOT http://media.blizzard.com
     * Do not call this function directly.
     * @param string $host
     * @return	boolean
     */
    private static function isHostAllowed($host) {
        return (in_array($host, self::$allowd_orign_hosts)) ? true : false;
    }

    /**
     * check if the file extension is allowed.
     * e.g. /path/to/file.extension
     * Do not call this function directly.
     * @param string $path
     * @return	boolean
     */
    private static function isTypeAllowed($path) {
        return (in_array(pathinfo($path, PATHINFO_EXTENSION), self::$allowed_assets)) ? true : false;
    }

    /**
     * Download a File to local asset dir.
     * Checks if a file exists. 
     * Returns the new path of the file.
     * @param string $url
     * @return	string
     */
    public static function download($url) {
        $asset = parse_url($url);
        if (empty($asset)) {
            throw new SystemException('invalid asset URL: '. $url);
        }
        if (self::isHostAllowed($asset['host']) && self::isTypeAllowed($asset['path'])) {
            $newfile = self::LOCAL_ASSET_DIR . pathinfo($asset['path'], PATHINFO_BASENAME);
            if (!file_exists($newfile)) static::save($url, $newfile);
            return $newfile;
        } else {
            throw new SystemException('host or extension not allowd: '. $url);
        }
    }
}    